%% Step 1: SRSE
% --- Initialize figure and variables --- %
clear all;
close all;

% Define paths
parentFolderPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables\MGGA\Emission Sources\Single_Point\Quantified_SC5';
outputFolder = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables';
outputFile = fullfile(outputFolder, 'SRSP_multiple_degrees_figure_SC6.png');

% Define experimental conditions
timeIntervals = {'5 minutes','15 minutes','30 minutes'};
degreeFolders = {'5 degrees','10 degrees','20 degrees'};

% Initialize results table for manuscript (using cell arrays for mixed types)
resultCell = cell(length(degreeFolders)*length(timeIntervals), 6);
header = {'WindAngle', 'TimeInterval', 'Slope', 'Slope_pValue', 'AdjR2', 'n'};
tableRow = 1;

% Create figure
fig = figure;
set(fig, 'Position', [100, 100, 1800, 1200]);

% Main analysis loop
for d = 1:length(degreeFolders)
    currentDegree = degreeFolders{d};
    
    for t = 1:length(timeIntervals)
        folderPath = fullfile(parentFolderPath, timeIntervals{t}, currentDegree);
        subplotIdx = (d-1)*length(timeIntervals) + t;
        subplot(3, 3, subplotIdx);

        % Check folder existence
        if ~isfolder(folderPath)
            warning('Folder missing: %s', folderPath);
            if d == 1, title(timeIntervals{t}, 'FontSize', 20); end
            continue;
        end

        % Load and process data
        files = dir(fullfile(folderPath, '*.csv'));
        [actualEmissionTemp, estimatedEmissionTemp] = deal([]);
        
        for i = 1:length(files)
            data = readtable(fullfile(folderPath, files(i).name));
            data.Actual_Emission_kg_h = 0.001 * 60 * 16.04 * 1 * data.C1FlowAvg_slpm / (0.082057 * 293.15);
            
            if all(ismember(["Actual_Emission_kg_h", "Emission_kg_h"], data.Properties.VariableNames))
                actualEmissionTemp = [actualEmissionTemp; data.Actual_Emission_kg_h(1)];
                estimatedEmissionTemp = [estimatedEmissionTemp; data.Emission_kg_h(1)];
            end
        end

        % Skip empty data
        if isempty(actualEmissionTemp) || isempty(estimatedEmissionTemp)
            if d == 1, title(timeIntervals{t}, 'FontSize', 20); end
            continue;
        end

        % --- Plotting --- %
        scatter(actualEmissionTemp, estimatedEmissionTemp, 70, 'o', 'MarkerEdgeColor', 'k', 'LineWidth', 2);
        hold on;
        
        % Formatting
if d == length(degreeFolders), xlabel('Actual (kg h^{-1})', 'FontSize', 20); end
if t == 1, ylabel(sprintf('\\bf{%s}\\rm\nEstimated (kg h^{-1})', currentDegree), 'FontSize', 24, 'Interpreter', 'tex'); end
if d == 1, title(timeIntervals{t}, 'FontSize', 24); end
grid off;
set(gca, 'FontSize', 24, 'XTick', 0:8);

        % Reference lines
        maxValue = max(actualEmissionTemp);
        xLine = linspace(0, maxValue, 100);
        plot(xLine, xLine, 'r--', 'LineWidth', 1.5); % 1:1 line
        fill([xLine, fliplr(xLine)], [xLine*0.7, fliplr(xLine*1.3)], 'k', 'FaceAlpha', 0.2, 'EdgeColor', 'none');

        % --- Regression Analysis --- %
        [p, S] = polyfit(actualEmissionTemp, estimatedEmissionTemp, 1);
        yFit = polyval(p, actualEmissionTemp);
        
        % Calculate statistics
        R = corrcoef(actualEmissionTemp, estimatedEmissionTemp);
        R_squared = R(1,2)^2;
        n = length(actualEmissionTemp);
        adjR2 = 1 - (1 - R_squared) * (n - 1) / (n - 2);
        
        % Calculate p-value for slope
        slope_SE = sqrt(diag((S.R)\inv(S.R'))./S.normr.^2./S.df);
        t_slope = p(1) / slope_SE(1);
        p_slope = 2 * tcdf(-abs(t_slope), S.df);
        
        % Store results
        resultCell{tableRow,1} = currentDegree;
        resultCell{tableRow,2} = timeIntervals{t};
        resultCell{tableRow,3} = p(1);
        resultCell{tableRow,4} = p_slope;
        resultCell{tableRow,5} = adjR2;
        resultCell{tableRow,6} = n;
        tableRow = tableRow + 1;

% Plot regression line
darkBlue = [0 0 0.5]; % RGB equivalent of #00008B
plot(xLine, polyval(p, xLine), 'Color', darkBlue, 'LineWidth', 2);

% Get current axis limits
currentXLim = xlim;
currentYLim = ylim;

% Display key metrics (top left, dark blue)
text(currentXLim(1) + 0.05*diff(currentXLim), currentYLim(2) - 0.05*diff(currentYLim), ...
    sprintf('y = %.2fx + %.2f\np = %.3f\nAdj. R² = %.2f\nn = %d', ...
    p(1), p(2), p_slope, adjR2, n), ...
    'FontSize', 20, 'Color', darkBlue, ...
    'VerticalAlignment', 'top', 'HorizontalAlignment', 'left');
    end
end

%sgtitle(sprintf('\\textbf{\\underline{Eddy Covariance}}\n\\textbf{\\underline{Single-Release Single-Point}}\\textbf{\\underline{- K\\&M Footprint}}'), ...
   % 'Interpreter', 'latex', 'FontSize', 18);

sgtitle(sprintf('\\textbf{\\underline{Gaussian Plume Inverse Method}}\n\\textbf{\\underline{Single-Release Single-Point-Sce. 6}}\\textbf{\\underline}'), ...
    'Interpreter', 'latex', 'FontSize', 20);
% Convert cell array to table and save
resultTable = cell2table(resultCell, 'VariableNames', header);
saveas(fig, outputFile);
writetable(resultTable, fullfile(outputFolder, 'regression_results.csv'));
disp(['Figure saved to: ', outputFile]);
disp('Regression results saved to regression_results.csv');

% Display the results table in command window
disp(' ');
disp('Regression Results:');
disp(resultTable);


%% Gaussian plots - MRSE



% --- Initialize figure and variables --- %
clear all;
close all;

% Define paths
parentFolderPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables\MGGA\Emission Sources\Multi_Point\Emission Point';
outputFolder = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables';
outputFile = fullfile(outputFolder, 'MRSP_multiple_degrees_figure.png');

% Define experimental conditions
timeIntervals = {'5 minutes','15 minutes','30 minutes'};
degreeFolders = {'5 degrees','10 degrees','20 degrees'};

% Initialize results table for manuscript (using cell arrays for mixed types)
resultCell = cell(length(degreeFolders)*length(timeIntervals), 6);
header = {'WindAngle', 'TimeInterval', 'Slope', 'Slope_pValue', 'AdjR2', 'n'};
tableRow = 1;

% Create figure
fig = figure;
set(fig, 'Position', [100, 100, 1800, 1200]);

% Main analysis loop
for d = 1:length(degreeFolders)
    currentDegree = degreeFolders{d};
    
    for t = 1:length(timeIntervals)
        folderPath = fullfile(parentFolderPath, 'Quantified_MRSP',timeIntervals{t}, currentDegree);
        subplotIdx = (d-1)*length(timeIntervals) + t;
        subplot(3, 3, subplotIdx);

        % Check folder existence
        if ~isfolder(folderPath)
            warning('Folder missing: %s', folderPath);
            if d == 1, title(timeIntervals{t}, 'FontSize', 20); end
            continue;
        end

        % Load and process data
        files = dir(fullfile(folderPath, '*.csv'));
        [actualEmissionTemp, estimatedEmissionTemp] = deal([]);
        
        for i = 1:length(files)
            data = readtable(fullfile(folderPath, files(i).name));
            data.Actual_Emission_kg_h = 0.001 * 60 * 16.04 * 1 * data.C1FlowAvg_slpm / (0.082057 * 293.15);
            
            if all(ismember(["Actual_Emission_kg_h", "Emission_kg_h"], data.Properties.VariableNames))
                actualEmissionTemp = [actualEmissionTemp; data.Actual_Emission_kg_h(1)];
                estimatedEmissionTemp = [estimatedEmissionTemp; data.Emission_kg_h(1)];
            end
        end

        % Skip empty data
        if isempty(actualEmissionTemp) || isempty(estimatedEmissionTemp)
            if d == 1, title(timeIntervals{t}, 'FontSize', 20); end
            continue;
        end

        % --- Plotting --- %
        scatter(actualEmissionTemp, estimatedEmissionTemp, 70, 'o', 'MarkerEdgeColor', 'k', 'LineWidth', 2);
        hold on;
        
        % Formatting
if d == length(degreeFolders), xlabel('Actual (kg h^{-1})', 'FontSize', 20); end
if t == 1, ylabel(sprintf('\\bf{%s}\\rm\nEstimated (kg h^{-1})', currentDegree), 'FontSize', 24, 'Interpreter', 'tex'); end
if d == 1, title(timeIntervals{t}, 'FontSize', 24); end
grid off;
set(gca, 'FontSize', 24, 'XTick', 0:8);

        % Reference lines
        maxValue = max(actualEmissionTemp);
        xLine = linspace(0, maxValue, 100);
        plot(xLine, xLine, 'r--', 'LineWidth', 1.5); % 1:1 line
        fill([xLine, fliplr(xLine)], [xLine*0.7, fliplr(xLine*1.3)], 'k', 'FaceAlpha', 0.2, 'EdgeColor', 'none');

        % --- Regression Analysis --- %
        [p, S] = polyfit(actualEmissionTemp, estimatedEmissionTemp, 1);
        yFit = polyval(p, actualEmissionTemp);
        
        % Calculate statistics
        R = corrcoef(actualEmissionTemp, estimatedEmissionTemp);
        R_squared = R(1,2)^2;
        n = length(actualEmissionTemp);
        adjR2 = 1 - (1 - R_squared) * (n - 1) / (n - 2);
        
        % Calculate p-value for slope
        slope_SE = sqrt(diag((S.R)\inv(S.R'))./S.normr.^2./S.df);
        t_slope = p(1) / slope_SE(1);
        p_slope = 2 * tcdf(-abs(t_slope), S.df);
        
        % Store results
        resultCell{tableRow,1} = currentDegree;
        resultCell{tableRow,2} = timeIntervals{t};
        resultCell{tableRow,3} = p(1);
        resultCell{tableRow,4} = p_slope;
        resultCell{tableRow,5} = adjR2;
        resultCell{tableRow,6} = n;
        tableRow = tableRow + 1;

% Plot regression line
darkBlue = [0 0 0.5]; % RGB equivalent of #00008B
plot(xLine, polyval(p, xLine), 'Color', darkBlue, 'LineWidth', 2);

% Get current axis limits
currentXLim = xlim;
currentYLim = ylim;

% Display key metrics (top left, dark blue)
text(currentXLim(1) + 0.5*diff(currentXLim), currentYLim(2) - 0.05*diff(currentYLim), ...
    sprintf('y = %.2fx + %.2f\np = %.3f\nAdj. R² = %.2f\nn = %d', ...
    p(1), p(2), p_slope, adjR2, n), ...
    'FontSize', 20, 'Color', darkBlue, ...
    'VerticalAlignment', 'top', 'HorizontalAlignment', 'left');
    end
end

%sgtitle(sprintf('\\textbf{\\underline{Eddy Covariance}}\n\\textbf{\\underline{Single-Release Single-Point}}\\textbf{\\underline{- K\\&M Footprint}}'), ...
   % 'Interpreter', 'latex', 'FontSize', 18);

sgtitle(sprintf('\\textbf{\\underline{Gaussian Plume Inverse Method}}\n\\textbf{\\underline{Multi-Release Single-Point}}\\textbf{\\underline}'), ...
    'Interpreter', 'latex', 'FontSize', 20);
% Convert cell array to table and save
resultTable = cell2table(resultCell, 'VariableNames', header);
saveas(fig, outputFile);
writetable(resultTable, fullfile(outputFolder, 'regression_results.csv'));
disp(['Figure saved to: ', outputFile]);
disp('Regression results saved to regression_results.csv');

% Display the results table in command window
disp(' ');
disp('Regression Results:');
disp(resultTable);


